<?php
/* --------------------------------------------------------------
   TrackingCodeCriteria.php 2020-03-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\TrackingCode\SqlCriteria;

use DateTime;
use Doctrine\DBAL\Query\QueryBuilder;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeSqlCriteria as TrackingCodeSqlCriteriaInterface;
use Gambio\Core\Criteria\AbstractSqlCriteria;

/**
 * Class TrackingCodeRestApiCriteria
 *
 * @package Gambio\Admin\TrackingCode\SqlCriteria
 */
class TrackingCodeSqlCriteria extends AbstractSqlCriteria implements TrackingCodeSqlCriteriaInterface
{
    /**
     * @var DateTime|null
     */
    private $createdAfter;
    
    
    /**
     * TrackingCodeSqlCriteria constructor.
     *
     * @param TrackingCodeFilters  $filters
     * @param TrackingCodeSortings $sortings
     * @param DateTime|null        $createdAfter
     */
    private function __construct(
        TrackingCodeFilters $filters,
        TrackingCodeSortings $sortings,
        ?DateTime $createdAfter
    ) {
        $this->filters      = $filters;
        $this->sortings     = $sortings;
        $this->createdAfter = $createdAfter;
    }
    
    
    /**
     * @param TrackingCodeFilters  $filters
     * @param TrackingCodeSortings $sortings
     * @param DateTime|null        $createdAfter
     *
     * @return TrackingCodeSqlCriteria
     */
    public static function create(
        TrackingCodeFilters $filters,
        TrackingCodeSortings $sortings,
        ?DateTime $createdAfter = null
    ): TrackingCodeSqlCriteria {
        return new self($filters, $sortings, $createdAfter);
    }
    
    
    /**
     * @param QueryBuilder $query
     */
    public function applyToQuery(QueryBuilder $query): void
    {
        parent::applyToQuery($query);
        
        if ($this->createdAfter !== null) {
            $query->andWhere('creation_date >= :criteria_created_after')
                ->setParameter(':criteria_created_after', $this->createdAfter->format('Y-m-d H:i:s'));
        }
    }
}